###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###
### CRC: DEMAND FOR OPEN TV                     ###*###*###*###*###*###*###*###*###
### DISCRETE CHOICE MODEL                       ###*###*###*###*###*###*###*###*###
### ESTIMATE PARAMETERS: R. COEF WITH SUPPLY    ###*###*###*###*###*###*###*###*###
###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###

# ____________________________________________________________________________ ####
# START ####

rm(list = ls())
library(plyr)
library(ivreg)
library(SQUAREM)
library(xtable)
library(Matrix)

# ____________________________________________________________________________ ####
# INPUTS ####

## Paths
inpath  <- '~/Data/Processed/'
outpath <- '~/Output/'
fnpath  <- '~/Code/'

# ____________________________________________________________________________ ####
# LOAD DATA ####

## Save sample
load(paste0(inpath, 'MDATA6.RData'))

## Functions
source(paste0(fnpath, 'functions_A.R'))

# ____________________________________________________________________________ ####
# ESTIMATION DATA ####

## Sort main data
DATA <- DATA[DATA$tb != 'E', ]
DATA <- DATA[order(DATA$t, DATA$chann, DATA$tb), ]

## Shares
S   <- ddply(DATA, .(t), mutate, x = Qp/sum(Qp)*1)$x
So  <- DATA$Spo
Sa  <- DATA$Sa
Sao <- DATA$Sao

## Observables
t  <- DATA$t
vn <- c('q', 'Dep', 'Hum', 'Not', 'Ser', 'Tvn')[-1]
q  <- DATA$tq
X  <- as.matrix(DATA[, vn])

## Brands and market effects
XF <- model.matrix(lm(p ~ chann, data = DATA))[, -1]
XB <- model.matrix(lm(p ~ tb, data = DATA))[, -1]
Xb <- model.matrix(lm(p ~ j, data = DATA))[, -1]
XT <- model.matrix(lm(p ~ as.character(y), data = DATA))[, -1]
Xt <- model.matrix(lm(p ~ t, data = DATA))[, -1]

## Initials
J      <- nrow(DATA)
NS     <- 5000
mkti   <- match(DATA$t, unique(DATA$t))
delta0 <- log(Sa) - log(Sao)

## Random shocks
set.seed(123)
v <- rnorm(NS * 2)
v <- matrix(v, 2, NS, byrow = T)

# ____________________________________________________________________________ ####
# ESTIMATE ####

## Instruments
ivn  <- c('zsp1', 'zsp2', 'zsp3', 'zsp4', 'zsp5', 'zsp6',  
          'zq1', 'zq2', 'zq3', 'zq4')
i    <- c(2:8, 10)#2:length(ivn)#
Z    <- as.matrix(DATA[, ivn[i]])
Z    <- cbind(Z, 'constant' = 1, X, XB, XF, XT); c(ncol(Z), rankMatrix(Z)[1])
A    <- solve(t(Z) %*% Z, tol = 1e-30)
XPF  <- cbind('constant' = 1, S, q, X, XB, XT); ivreg(delta0 ~ 0 + XPF, ~ Z)
ZPZ  <- Z %*% A %*% t(Z)
XZPZ <- solve(t(XPF) %*% ZPZ %*% XPF) %*% t(XPF) %*% ZPZ

# Check identification range
zeta0 <- c(0.01, 0.01)#abs(ans$par)
rankMatrix(Z)[1] >= ncol(XPF) + length(zeta0) 

## First stage
ans1 <- optim(zeta0, gmm, method = "Nelder-Mead", control = list(maxit = 10000))
save(ans1, file = paste0(outpath, 'EstimA_06_01.RData'))

zeta  <- abs(ans1$par)
out   <- struct(zeta)
si    <- out$si
b     <- out$b; b
thi   <- b[2] + zeta[1]*v[1, ]; summary(thi)
lmi   <- b[3] + zeta[2]*v[2, ]; summary(lmi)

## First-Stage Results
zeta1 <- abs(ans1$par)
OUT   <- struct(zeta1)
om    <- OUT$xi
A2    <- 0
for(i in 1:nrow(Z)) A2 <- A2 + om[i]^2 * Z[i, ]%*%t(Z[i, ])/nrow(Z)
A2    <- solve(A2, tol = 1e-30)
ZPZ   <- Z %*% A2 %*% t(Z)

## Second Stage Estimation
ans  <- optim(zeta1, gmm, method = "Nelder-Mead", control = list(maxit = 15000))
save(ans, file = paste0(outpath, 'EstimA_06_02.RData'))

## Results
load(paste0(outpath, 'EstimA_06_02.RData'))
zeta  <- abs(ans$par)
out   <- struct(zeta)
delta <- out$delta
si    <- out$si
b     <- out$b; b
thi   <- b[2] + zeta[1]*v[1, ]; summary(thi)
lmi   <- b[3] + zeta[2]*v[2, ]; summary(lmi)
xi    <- out$xi
save(list = c('b', 'si', 'thi', 'lmi', 'delta', 'XPF', 'xi', 'zeta'), 
     file = paste0(outpath, 'outsA_06.RData'))

# ____________________________________________________________________________ ####
# VARIANCE ####

## Compute jacobian and moment covariance 
om       <- xi
Z2       <- Z
theta    <- c(zeta, b)
eps      <- 1e-13
Kg       <- ncol(Z)
Kb       <- length(theta)
dtheta   <- numeric(Kb)
G <- PSI <- 0
for(i in 1:nrow(Z2)) {
  
  print(i)
  # Jacobian (Kg x Kb)
  Gi <- matrix(0, Kg, Kb)
  for(k in 1:Kb) {
    dtheta[k] <- eps
    if(k <= 2) {
      g0 <- Z2[i, ] * om[i]
      g1 <- Z2[i, ] * omega2(zeta = zeta - dtheta[1:2])[i]
      g2 <- Z2[i, ] * omega2(zeta = zeta + dtheta[1:2])[i]
    } else {
      g0 <- Z2[i, ] * om[i]
      g1 <- Z2[i, ] * omega1(theta[-(1:2)] - dtheta[3:Kb])[i]
      g2 <- Z2[i, ] * omega1(theta[-(1:2)] + dtheta[3:Kb])[i]
    }
    Gi[, k] <- 0.5*(g1 - g0)/(-eps) + (g2 - g0)*0.5/eps
    dtheta  <- numeric(Kb)
  }
  G   <- G + Gi/nrow(Z) 
  # Moment covariance (Kg x Kg)
  PSI <- PSI + om[i]^2 * Z2[i, ]%*%t(Z2[i, ])/nrow(Z2)
  
}

## Weight matrix (Kg x Kg)
W <- A2

## Parameter covariance matrix
A   <- t(G) %*% W %*% G
A   <- solve(A, tol = 1e-20)
B   <- t(G) %*% W %*% PSI %*% W %*% G
Var <- (A %*% B %*% A) / (nrow(Z2) - Kb)
se  <- sqrt((diag(Var)))
save(Var, file = paste0(outpath, 'VCOVA_06.RData'))

# ____________________________________________________________________________ ####
# HYPOTHESIS TESTS ####

## Coefficients' significancy
sb <- tail(se, length(b))
pv <- (1 - pt(abs(b/sb), df = J - Kb))/2
x <- cbind(b, sb, pv)
row.names(x)[1:11] <- c('Constante', 'S', 'q', 'Deportivo',
  'Humor', 'Noticiero', 'Series', 'Tv-Novela', 'Franja-B',
  'Franja-C', 'Franja-D')
print(xtable(x, digits = 3))

# ____________________________________________________________________________ ####
# END ####