###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###
### CRC: DEMAND FOR OPEN TV                     ###*###*###*###*###*###*###*###*###
### DISCRETE CHOICE MODEL                       ###*###*###*###*###*###*###*###*###
### ESTIMATE PARAMETERS: R. COEF WITH SUPPLY    ###*###*###*###*###*###*###*###*###
###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###

# ____________________________________________________________________________ ####
# START ####

rm(list = ls())
library(plyr)
library(ivreg)
library(reshape2)
library(SQUAREM)
library(xtable)
library(Matrix)

# ____________________________________________________________________________ ####
# INPUTS ####

## Paths
inpath  <- '~/Data/Processed/'
outpath <- '~/Output/'
fnpath  <- '~/Code/'

# ____________________________________________________________________________ ####
# LOAD DATA ####

## Save sample
load(paste0(inpath, 'MDATA6.RData'))

## Functions
source(paste0(fnpath, 'functions_P2.R'))

# ____________________________________________________________________________ ####
# ESTIMATION DATA ####

## Sort main data
DATA <- DATA[DATA$tb != 'E', ]
DATA <- DATA[order(DATA$t, DATA$chann, DATA$tb), ]

## Price, discount and quality
p  <- DATA$p
D  <- DATA$D
q  <- DATA$q

## Shares
S  <- DATA$Sp
So <- DATA$Spo
Sa <- ddply(DATA, .(t), mutate, x = Qa/sum(Qa)*1)$x#DATA$Sa

## Observables
t  <- DATA$t
vn <- c('Dep', 'Hum', 'Not', 'Ser', 'Tvn')
X  <- as.matrix(DATA[, vn])

## Brands and market effects
XF  <- model.matrix(lm(p ~ chann, data = DATA))[, -1]
XBr <- model.matrix(lm(p ~ 0 + tb, data = DATA))
XB  <- XBr[, -1]
Xb  <- model.matrix(lm(p ~ j, data = DATA))[, -1]
XT  <- model.matrix(lm(p ~ as.character(y), data = DATA))[, -1]
Xt  <- model.matrix(lm(p ~ t, data = DATA))[, -1]

## Initials
J      <- nrow(DATA)
NS     <- 5000
mkti   <- match(DATA$t, unique(DATA$t))
delta0 <- log(S) - log(So)

## Firm matrix
Tn <- length(unique(DATA$t))
H  <- matrix(0, ncol = J, nrow = J)
m  <- unique(DATA$t)
for(tt in 1:Tn) {
  x <- cbind('channCARACOL' = 1 - rowSums(XF[t == m[tt],]), XF[t == m[tt],])
  x <- x %*% t(x)
  i <- which(mkti == tt)
  H[i, i] <- x
}

## Random shocks
set.seed(123)
v <- rnorm(NS * 2)
v <- matrix(v, 2, NS, byrow = T)

load(paste0(outpath, 'outsA_06.RData'))
xia <- xi
deltaa <- delta
XPFa <- XPF
ba <- b
zetaa <- zeta

# ____________________________________________________________________________ ####
# ESTIMATE ####

## Instruments
ivn  <- c(paste0('zp', 1:6), paste0('zD', 1:6), paste0('zs', 1:6))
i    <- c(1:2, 3:8, 10:12, 15:length(ivn))#1:length(ivn)#
Z    <- as.matrix(DATA[, ivn[i]])
Z    <- cbind(Z, 1, X, XB, XF, XT); c(ncol(Z), rankMatrix(Z)[1])
A    <- solve(t(Z) %*% Z, tol = 1e-30)
XPF  <- cbind('constant' = 1, Sa, X, XB, XF); ivreg(delta0 ~ 0 + XPF, ~ Z)
WPF  <- cbind('constant' = 1, XF, XB, XT)
ZPZ  <- Z %*% A %*% t(Z)
ZPZ2 <- rbind(Z, Z, Z) %*% A %*% t(rbind(Z, Z, Z))
XZPZ <- solve(t(XPF) %*% ZPZ %*% XPF) %*% t(XPF) %*% ZPZ
WZPZ <- solve(t(WPF) %*% ZPZ %*% WPF) %*% t(WPF) %*% ZPZ

# Check identification range
load(paste0(outpath, 'Estim_06_01.RData'))
zeta0 <- cons(ans1$par[c(1, rep(2, 4), 3:4)])
rankMatrix(Z)[1] >= ncol(XPF) + ncol(WPF) + length(zeta0) 

## First stage
ans1 <- optim(zeta0, gmm, method = "Nelder-Mead", control = list(maxit = 10000))
save(ans1, file = paste0(outpath, 'Estim_06_01b.RData'))

zeta <- cons(ans1$par)
out  <- struct(zeta)
b    <- out$b; b
ai   <- matrix(zeta[1], J, NS) + zeta[3]*matrix(v[1, ], J, NS, byrow = T); summary(as.numeric(ai))
cm   <- out$cm; summary(cm); summary(p*(1 - D) - cm)

## First-Stage Results
zeta1 <- cons(ans1$par)
OUT   <- struct(zeta1)
om    <- c(OUT$xi, OUT$wo, OUT$zt)
A2    <- 0
for(i in 1:nrow(Z)) A2 <- A2 + om[i]^2 * Z[i, ]%*%t(Z[i, ])/(nrow(Z)*3)
for(i in 1:nrow(Z)) A2 <- A2 + om[i + nrow(Z)]^2 * Z[i, ]%*%t(Z[i, ])/(nrow(Z)*3)
for(i in 1:nrow(Z)) A2 <- A2 + om[i + 2*nrow(Z)]^2 * Z[i, ]%*%t(Z[i, ])/(nrow(Z)*3)
A2    <- solve(A2, tol = 1e-30)
ZPZ   <- Z %*% A2 %*% t(Z)
ZPZ2  <- rbind(Z, Z, Z) %*% A2 %*% t(rbind(Z, Z, Z))

## Second Stage Estimation
ans  <- optim(zeta1, gmm, method = "Nelder-Mead", control = list(maxit = 15000))
save(ans, file = paste0(outpath, 'Estim_06_02b.RData'))

## Results
load(paste0(outpath, 'Estim_06_02b.RData'))
zeta  <- cons(ans$par)
out   <- struct(zeta)
delta <- out$delta
cm    <- out$cm; summary(cbind('cm' = cm, 'mkp1' = p - cm, 'mkp2' = p*(1 - D) - cm))
si    <- out$si
b     <- out$b; b
ai    <- zeta[1] + zeta[6]*v[1, ]; summary(ai)
rho   <- zeta[2:5]
phi   <- b[2] + zeta[7]*v[2, ]; summary(phi)
g     <- out$g; g
xi    <- out$xi
wo    <- out$wo
zt    <- out$zt

x <- data.frame(DATA, cm = cm)#[cm >= 0, ]
x <- ddply(x, .(y, chann), summarize, v = weighted.mean(cm, Qp))
x <- dcast(x, chann ~ y, value.var = 'v')
x <- xtable(x)
print.xtable(x, include.rownames = F)

x <- data.frame(DATA, cm = cm)
x <- ddply(x, .(y, chann), summarize, v = weighted.mean(1 - cm/p, Qp)*100)
x <- dcast(x, chann ~ y, value.var = 'v')
x <- xtable(x)
print.xtable(x, include.rownames = F)

x <- data.frame(DATA, cm = cm)
x <- ddply(x, .(y, chann), summarize, v = weighted.mean((p*(1 - D) - cm)/p, w = Qp)*100)
x <- dcast(x, chann ~ y, value.var = 'v')
x <- xtable(x)
print.xtable(x, include.rownames = F)

# ____________________________________________________________________________ ####
# VARIANCE ####

## Compute jacobian and moment covariance 
om       <- c(xi)#, wo, zt)
Z2       <- rbind(Z)#, Z, Z)
theta    <- c(zeta, b)
eps      <- 1e-13
Kg       <- ncol(Z)
Kb       <- length(theta)
dtheta   <- numeric(Kb)
G   <- PSI <- 0
OM1 <- OM2 <- matrix(0, length(om), Kb)
for(k in 1:Kb) {
  dtheta[k] <- eps
  if(k <= length(zeta)) {
    om1 <- omega2(zeta = zeta - dtheta[1:length(zeta)])[1:nrow(Z2)]
    om2 <- omega2(zeta = zeta + dtheta[1:length(zeta)])[1:nrow(Z2)]
  } else {
    om1 <- omega1(theta[-(1:length(zeta))] - dtheta[(length(zeta) + 1):Kb])[1:nrow(Z2)]
    om2 <- omega1(theta[-(1:length(zeta))] + dtheta[(length(zeta) + 1):Kb])[1:nrow(Z2)]
  }
  dtheta  <- numeric(Kb)
  OM1[, k] <- om1
  OM2[, k] <- om2
}

for(i in 1:nrow(Z2)) {
  print(i)
  # Jacobian (Kg x Kb)
  Gi <- matrix(0, Kg, Kb)
  for(k in 1:Kb) {
    g0 <- Z2[i, ] * om[i]
    g1 <- Z2[i, ] * OM1[i, k]
    g2 <- Z2[i, ] * OM2[i, k]
    Gi[, k] <- 0.5*(g1 - g0)/(-eps) + (g2 - g0)*0.5/eps
  }
  G   <- G + Gi/nrow(Z2) 
  # Moment covariance (Kg x Kg)
  PSI <- PSI + om[i]^2 * Z2[i, ]%*%t(Z2[i, ])/nrow(Z2)
}

## Weight matrix (Kg x Kg)
W <- A2
## Parameter covariance matrix
A   <- t(G) %*% W %*% G
A   <- solve(A, tol = 1e-20)
B   <- t(G) %*% W %*% PSI %*% W %*% G
Var <- (A %*% B %*% A) / (J - Kb)
save(Var, file = paste0(outpath, 'VCOVP_06b.RData'))

# ____________________________________________________________________________ ####
# HYPOTHESIS TESTS ####

## Coefficients' significancy
load(paste0(outpath, 'VCOVP_06b.RData'))
bb <- c(zeta[1:5], b)
se  <- sqrt((diag(Var)))
sb <- tail(se, length(bb)); #sb[1] <- sb[1]/300
pv <- (1 - pt(abs(bb/sb), df = J - Kb))/2
x  <- cbind(bb, sb, pv)
row.names(x) <- c('p', paste0('D', LETTERS[1:4]), 
                  c('Constante', 'Sa', 'Deportivo', 'Humor', 'Noticiero', 
                    'Serries', 'Tv-Novela', 'Franja-B', 'Franja-C', 
                    'Franja-D', 'RCN', 'UNO'))
print(xtable(x[c(6, 1:5, 7:nrow(x)), ], digits = 3))

# ____________________________________________________________________________ ####
# PRICE ELASTICITIES ####

## Price Elasticities
aim  <- matrix(zeta[1], J, NS) + zeta[6]*matrix(v[1, ], J, NS, byrow = T)
dsdp <- -(aim * si) %*% t(si) / NS            # ds_r/dp_j
diag(dsdp) <- rowMeans(aim * si * (1 - si))   # ds_j/dp_j (r = j)
eta  <- dsdp * ((1 / S) %*% t(p)) 
etap <- 0
w <- ddply(DATA, .(t), summarize, w = sum(Qp))$w; w <- w/sum(w)
mkts <- unique(DATA$t)
channs <- unique(DATA$chann)
for(tt in 1:length(w)) {
  print(tt)
  i <- which(mkts[tt] == DATA$t)
  etai  <- eta[i, i]
  etac <- matrix(NA, 3, 3)
  for(j in channs) {
    for(k in channs) {
      m <- which(DATA$chann[i] == j)
      n <- which(DATA$chann[i] == k)
      wa <- DATA$Qp[i][m]; wa <- wa/sum(wa)
      wb <- DATA$Qp[i][n]; wb <- wb/sum(wb)
      etaimn <- etai[m, ][, n]
      etac[which(channs == j), which(channs == k)] <- t(wa) %*% etaimn %*% wb
    }
  }
  etap <- etap + etac*w[tt]
}
row.names(etap) <- colnames(etap) <- unique(DATA$chann)
x <- xtable(etap)
print.xtable(x, include.rownames = T)

# END ####