#' *****************************************************************************
#'                  Agrupación de datos Calidad de servicio móvil
#'                              Abril 2024 
#' *****************************************************************************


#1. ----------------------preparación del ambiente------------------------------
# limpieza del ambiente
rm(list = ls())

# carga de librerías
require(pacman)

p_load(ggplot2, #visualización de datos a través de gráficos
       factoextra, #extraer y visualizar resultados de análisis multivariado
       dplyr, #manipulación de datos
       tidyr, #datos ordenados y desordenados
       readxl, #lectura de archivos de excel
       tidyverse, #instalar y cargar librerías de tidyverse
       cluster, #encontrar los grupos en los datos
       NbClust, #determinar el número óptimo de clústers
       writexl, #exportar a xlsx
       gridExtra, #funciones para trabajar con gráficos 
       magrittr, #canalización de comandos
       reshape2, #cambio de formato de una base de datos
       clValid) #validación de resultados del cluster

# establecer directorio de trabajo
path_script <- rstudioapi::getActiveDocumentContext()$path
path_folder <- dirname(path_script)
setwd(path_folder)


#2. --------------------------cargar los datos ---------------------------------
# datos de caracterización municipal
data_base <- read_excel("1_Caracterización_Municipal.xlsx") %>%
  mutate_all(as.numeric) 

# datos de calidad móvil a nivel municipal
calidad_base <- read_excel("2_Calidad_linea_base.xlsx") %>%
  rename(codmpio=ID_MUNICIPIO) #cambiar nombre de la variable de código de mpio

# Municipios excluidos de calidad
mpios_excluidos <- read_excel("3_320_Municipios_Exceptuados.xlsx") %>%
  rename(codmpio=DIVIPOLA) %>% #cambiar nombre de la variable de código de mpio
  subset(select=c(codmpio,Exceptuados)) # mantener solo dos variables

# Nombres de deparatamentos y municipios
divipola <- read_excel("4_Divipola.xlsx")

#3. ----------------------unión de base de datos--------------------------------

# unir caracterización y calidad
data_base <- left_join(data_base,
                       calidad_base,
                       by="codmpio")

# unir con excluidos y filtrar
data_base <- left_join(data_base, 
                       mpios_excluidos, 
                       by="codmpio") %>% #unión de bases de datos 
  mutate(Exceptuados= ifelse(is.na(Exceptuados), 0,
                             Exceptuados)) %>% #agrerar cero a los no excluidos
  filter(Exceptuados == 0) #filtrar los no exlcuidos
  

# valores perdidos
colSums(is.na(data_base))
prop.table(colSums(is.na(data_base)))

#mantener las observaciones completas. 
data_base = data_base[complete.cases(data_base),]


#4.-----------------------preparación de los datos------------------------------

#keep municipalities code
Y <- data_base %>% 
  select(c("codmpio")) 

#all variables as numerics
X <- data_base %>%
  select(-c("codmpio",
            "Exceptuados", 
            "TestCount_DL", 
            "TestCount_UL", 
            "TestCount_PL", 
            "TestCount_PG", 
            "TestCount_JT")) %>% #Mantener solo variables de interés)) %>%
  mutate_all(as.numeric)

# estandarizar las variables, centrándolas y escalandolas con respecto a la 
# desviación estándar
X_scale<-scale(X)

# Medias de las variables después de escalar los datos
medias <- colMeans(X)

# Calcular la desviación estándar de cada variable en X
desviacion_estandar <- apply(X, 2, sd)

#5.---------------------------análisis gráfico----------------------------------


#total sums of squares method within cluster
graphic_1 <- fviz_nbclust(X_scale, pam, method = "wss", k.max = 10) +
  geom_vline(xintercept = 8, linetype = 2) +
  xlab("Número de Clusters K") +
  ylab("Suma Cuadrada Total Intracluster") +
  ggtitle("Número Óptimo de clúster") +
  theme(plot.title = element_text(size = 14),   #modify size
        axis.title.x = element_text(size = 11), #modify X axis size
        axis.title.y = element_text(size = 11), #modify Y axis size
        axis.text.x = element_text(size = 10),  #modify X axis values size
        axis.text.y = element_text(size = 10))  #modify Y axis values size
print(graphic_1)

#gap statistic method - Hastie (2001)
gap_stat <- clusGap (X_scale,
                     FUNcluster = pam,
                     K.max = 10, # clústeres máximos a considerar 
                     B = 10) # iteraciones totales de arranque
gap_stat[["Tab"]]

# gráfico de gap statistic
graphic_2 <- fviz_gap_stat(gap_stat) +
  geom_vline(xintercept = 3, linetype = 2) +
  ggtitle("Brecha Estadística") + 
  xlab("Numero de Clusters K") +
  ylab("Gap statistic (k)") + 
  theme(plot.title = element_text(size = 14),   #modify size
        axis.title.x = element_text(size = 11), #modify X axis size
        axis.title.y = element_text(size = 11), #modify Y axis size
        axis.text.x = element_text(size = 10),  #modify X axis values size
        axis.text.y = element_text(size = 10))  #modify Y axis values size
print(graphic_2)

# unir las dos gráficas
graficas_ <- grid.arrange(graphic_1, graphic_2, ncol = 2)
ggsave("Pruebas_Cluster.png", graficas_)

#6.---------------------------pruebas estadísticas------------------------------ 

# para correr se debe confirmar con la letra "y" 
comparacion_clusters <- clValid(
  obj        = X_scale,
  nClust     = 3:10,
  clMethods  = c("pam","kmeans"),
  validation = c("stability", "internal")
)

summary(comparacion_clusters)

#7.------------------------segementación de los cluster-------------------------
#aplicación del proceso de 3 k-medioides
set.seed(42)
kmed <- pam(X_scale,k=3)

# Acceder a los centroides
medoides <- kmed$medoids

# Acceder a la asignación de clúster para cada observación
cluster_asignado <- kmed$clustering

# Crear un nuevo data frame con los medoides y la asignación de clúster
medoides_con_cluster <- cbind(medoides, Clúster = cluster_asignado)

medoides_con_cluster <- as.data.frame(medoides_con_cluster)


# Medias de las variables después de escalar los datos
medias <- colMeans(X)

# Calcular la desviación estándar de cada variable en X
desviacion_estandar <- apply(X, 2, sd)

# Crear un data frame con los nombres de las variables, sus medias y 
# desviaciones estándar
estadisticos_df <- data.frame(indicador = names(medias), 
                              media = medias, 
                              desviacion = desviacion_estandar)
print(estadisticos_df)


#8.-------------------Estadísticas descriptivas de los grupos-------------------

#step 2: collapse each variable by cluster, calculating means and weighted means
tabla_estadisticas <- tabla_estadisticas %>%
  group_by(medoids_clusters) %>%
  summarise(
    #counts observations in each group
    num_obs = n(),
    #calculate weighted average download speed
    Descarga = mean(Descarga),
    #calculate weighted average upload speed
    Carga = mean(Carga),
    #calculate weighted average Jitter
    Jitter = mean(as.numeric(Jitter)),
    #calculate weighted average ping
    Latencia = mean(latencia),
    #calculate weighted average packet loss
    Perdida_paquetes = mean(Perdida_de_paquetes),
    #calculate average 4G penetration
    Ind_penetracion_4G = mean(INDICADOR_LTE),
    #calculate average 4G network capacity
    CAPACIDAD_4G_HAB = mean(Cap_4g_pc),
    #calculate the average rural proportion
    Pob_rural = mean(CEN_POB_RURAL_DIS),
    #calculate average population density
    Dens_POB = mean(DENS_POB),
    #average of Índice de Pobreza Multidimensional (IPM)
    IPM = mean(IPM),
    #calculate distance to capital means
    distancia_capital = mean(DIST_CAPITAL),
    #calculate the number of average suppliers
    Num_provedores = mean(NUM_PROV),
    #calculate the average number of mobile service lines
    Lineas_servicio = mean(lineas_en_serv_promedio)
  ) %>%
  #step 3: organize data in wide format
  gather(key = "variable", "value", num_obs, Descarga, Carga, Perdida_paquetes, 
         Jitter, Latencia, Ind_penetracion_4G, CAPACIDAD_4G_HAB,Pob_rural, IPM, 
         distancia_capital, Num_provedores, Lineas_servicio, Dens_POB) %>%
  spread(medoids_clusters, value) %>%  group_by(variable)

#9.-------------------exportar estadísticas descriptivas------------------------

#round numeric columns to 2 decimal places
tabla_estadisticas <- tabla_estadisticas %>%
  mutate(across(where(is.numeric),
                ~round(., 2)))
#export summary table
write_xlsx(tabla_estadisticas, "summary_table_by_cluster.xlsx")

#10.------------------------asignación de clúster a mpios------------------------


writexl::write_xlsx(medoides_con_cluster, "Medoides.xlsx")
writexl::write_xlsx(X_scale, "X_scale.xlsx")
writexl::write_xlsx(estadisticos_df, "estadisticos_df.xlsx")

# Unión y filtrado de datos
asignacion_cluster <- calidad_base %>%
  left_join(mpios_excluidos, by = "codmpio") %>%
  mutate(Exceptuados = if_else(is.na(Exceptuados), 0, Exceptuados)) %>%
  filter(Exceptuados == 0) %>% #eliminación de municipios excluidos
  anti_join(Y, by = "codmpio") %>% #filtrar los 32 municipios 
  mutate_all(as.numeric)

# Reshape del dataframe
estadisticos_df_wide <- estadisticos_df %>%
  pivot_wider(names_from = indicador, values_from = c(media, desviacion))

# Variables del dataframe
variables <- c("Descarga", "Carga", "Perdida_de_paquetes", "latencia", "Jitter")

# Normalización de variables
mpios_incluidos <- calidad_base %>%
  right_join(Y, by = "codmpio") %>%
  mutate(Cluster = kmed$clustering) %>%
  select(c("codmpio", 
           "Descarga", 
           "Carga", 
           "Perdida_de_paquetes", 
           "latencia", 
           "Jitter",
           "Cluster")) %>%
  mutate_all(as.numeric)

# Calcular los mínimos y máximos para las variables
min_max <- mpios_incluidos %>%
  summarise(across(c(Descarga, Carga, Perdida_de_paquetes, latencia, Jitter), 
                   list(min = min, max = max)))

# Normalizar las variables
estandarizado_df <- asignacion_cluster %>%
  mutate(across(c(Descarga, Carga, Perdida_de_paquetes, latencia, Jitter), 
                ~ (.- min_max[[paste0(cur_column(), "_min")]]) / 
                  (min_max[[paste0(cur_column(), "_max")]] - 
                     min_max[[paste0(cur_column(), "_min")]]), 
                .names = "{.col}_norm")) %>%
  select(codmpio, ends_with("_norm"))

pesos_calidad <- list(
  Descarga = 0.534364,
  Carga = 0.465636
)

# Multiplicar los valores normalizados por los pesos correspondientes
estandarizado_df <- estandarizado_df %>%
  mutate(Suma_Ponderada = (Carga_norm * pesos_calidad[["Carga"]]) + 
           (Descarga_norm * pesos_calidad[["Descarga"]]))

# Reshape del dataframe
estadisticos_df_wide <- estadisticos_df %>%
  pivot_wider(names_from = indicador, values_from = c(media, desviacion))

#step 1: add cluster results to data base in a new table
data_base <- data_base %>% 
  mutate(medoids_clusters = kmed$clustering)

estadisticos_df_wide$media_Descarga_1 <-  mean(filter(data_base, 
                                                      medoids_clusters == 1)$Descarga)
estadisticos_df_wide$media_Descarga_2 <-  mean(filter(data_base, 
                                                      medoids_clusters == 2)$Descarga)
estadisticos_df_wide$media_Descarga_3 <-  mean(filter(data_base, 
                                                      medoids_clusters == 3)$Descarga)


estadisticos_df_wide$media_Carga_1 <-  mean(filter(data_base, 
                                                   medoids_clusters == 1)$Carga)
estadisticos_df_wide$media_Carga_2 <-  mean(filter(data_base, 
                                                   medoids_clusters == 2)$Carga)
estadisticos_df_wide$media_Carga_3 <-  mean(filter(data_base, 
                                                   medoids_clusters == 3)$Carga)


# Normalizar los centroides cluster 1
estadisticos_df_wide <- estadisticos_df_wide %>%
  mutate(media_Descarga_norm_1 = (media_Descarga_1 - min_max$Descarga_min)/(min_max$Descarga_max - min_max$Descarga_min)) %>%
  mutate(media_Carga_norm_1 = (media_Carga_1 - min_max$Carga_min)/(min_max$Carga_max - min_max$Carga_min))

# Normalizar los centroides cluster 2
estadisticos_df_wide <- estadisticos_df_wide %>%
  mutate(media_Descarga_norm_2 = (media_Descarga_2 - min_max$Descarga_min)/(min_max$Descarga_max - min_max$Descarga_min)) %>%
  mutate(media_Carga_norm_2 = (media_Carga_2 - min_max$Carga_min)/(min_max$Carga_max - min_max$Carga_min))

# Normalizar los centroides cluster 3
estadisticos_df_wide <- estadisticos_df_wide %>%
  mutate(media_Descarga_norm_3 = (media_Descarga_3 - min_max$Descarga_min)/(min_max$Descarga_max - min_max$Descarga_min)) %>%
  mutate(media_Carga_norm_3 = (media_Carga_3- min_max$Carga_min)/(min_max$Carga_max - min_max$Carga_min))


estadisticos_df_wide <- estadisticos_df_wide %>%
  mutate(Suma_Ponderada_centroides_1 := (media_Carga_norm_1 * pesos_calidad[["Carga"]]) + 
           (media_Descarga_norm_1 * pesos_calidad[["Descarga"]]))

estadisticos_df_wide <- estadisticos_df_wide %>%
  mutate(Suma_Ponderada_centroides_2 := (media_Carga_norm_2 * pesos_calidad[["Carga"]]) + 
           (media_Descarga_norm_2 * pesos_calidad[["Descarga"]]))

estadisticos_df_wide <- estadisticos_df_wide %>%
  mutate(Suma_Ponderada_centroides_3 := (media_Carga_norm_3 * pesos_calidad[["Carga"]]) + 
           (media_Descarga_norm_3 * pesos_calidad[["Descarga"]]))

estandarizado_df <- estandarizado_df %>%
  mutate(Suma_Ponderada_centroides_1 = estadisticos_df_wide$Suma_Ponderada_centroides_1) %>%
  mutate(Suma_Ponderada_centroides_2 = estadisticos_df_wide$Suma_Ponderada_centroides_2) %>%
  mutate(Suma_Ponderada_centroides_3 = estadisticos_df_wide$Suma_Ponderada_centroides_3)



estandarizado_df <- estandarizado_df %>%
  mutate(Diff_Cluster_1 = estandarizado_df$Suma_Ponderada - estandarizado_df$Suma_Ponderada_centroides_1) %>%
  mutate(Diff_Cluster_2 = estandarizado_df$Suma_Ponderada - estandarizado_df$Suma_Ponderada_centroides_2) %>%
  mutate(Diff_Cluster_3 = estandarizado_df$Suma_Ponderada - estandarizado_df$Suma_Ponderada_centroides_3)

# Cálculo de diferencias
for (i in 1:3) {
  nombre_columna <- paste("Diff_Cluster", i, sep = "_")
  estandarizado_df[[nombre_columna]] <- abs(estandarizado_df$Suma_Ponderada - estandarizado_df[[paste0("Suma_Ponderada_centroides_", i)]])
}


# Asignación de cluster de carga y descarga
estandarizado_df <- estandarizado_df %>%
  mutate(Cluster_CyC = case_when(
    Diff_Cluster_1 < Diff_Cluster_2 & 
      Diff_Cluster_1 < Diff_Cluster_3 ~ 1,
    Diff_Cluster_2 < Diff_Cluster_1 & 
      Diff_Cluster_2 < Diff_Cluster_3 ~ 2,
    TRUE ~ 3
  ))

#11.-----exportar la lista final del cluster, excluidos y sin información-------

cluster_results <- data_base %>% 
  mutate(Cluster = kmed$clustering) %>% #add cluster results to data base
  subset(select=c(codmpio, Cluster)) #select variable to joint. 

#joint municipalities to cluster
cluster_results <- left_join(divipola, 
                      cluster_results, 
                      by="codmpio") %>% #
  subset(select=c(DP, DPNOM, codmpio, DPMP, Cluster))  

#joint cluster to excluded and not information (cod=99) municipalities. 
mpio_cluster_list <- left_join(cluster_results, 
                               mpios_excluidos, 
                               by="codmpio") %>% 
  mutate(Exceptuados = ifelse(is.na(Exceptuados), 0, Exceptuados),
         grupo_mpo = ifelse(!is.na(Cluster), Cluster, 
                            ifelse(Exceptuados == 1, 0, 99)))

estandarizado_df <- estandarizado_df %>%
  subset(select=c(codmpio, Cluster_CyC))

mpio_cluster_list <- left_join(mpio_cluster_list,
                               estandarizado_df, 
                               by="codmpio") %>%
  mutate(nivel_calidad = ifelse(Cluster == 1, "Alto",
                                ifelse(Cluster == 2, "Medio", 
                                       ifelse(Cluster == 3, "Bajo", NA)))) %>%
  mutate(nivel_calidad = ifelse(is.na(nivel_calidad) & Cluster_CyC == 1, "Alto",
                                ifelse(is.na(nivel_calidad) & Cluster_CyC == 2, "Medio",
                                       ifelse(is.na(nivel_calidad) & Cluster_CyC == 3, "Bajo", nivel_calidad)))) %>%
  mutate(nivel_calidad = ifelse(is.na(nivel_calidad) & Exceptuados == 1, "Exceptuados", nivel_calidad)) %>%
  mutate(nivel_calidad = ifelse(is.na(nivel_calidad) & grupo_mpo == 99, "Bajo", nivel_calidad)) %>%
  subset(select=c(DP, DPNOM, codmpio, DPMP, nivel_calidad))
                                       

table(mpio_cluster_list$nivel_calidad)

#export list:
writexl::write_xlsx(mpio_cluster_list, "mpio_cluster_list.xlsx")

