###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###
### CRC: DEMAND FOR MOBILE SERVICES             ###*###*###*###*###*###*###*###*###
### DISCRETE CHOICE MODEL                       ###*###*###*###*###*###*###*###*###
### ESTIMATE PARAMETERS                         ###*###*###*###*###*###*###*###*###
###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###*###

# ____________________________________________________________________________ ####
# START ####

rm(list = ls())
library(readxl)
library(plyr)
library(lubridate)
library(reshape2)
library(ivreg)
library(SQUAREM)
library(Matrix)
library(xtable)

# ____________________________________________________________________________ ####
# INPUTS ####

## Paths
inpath  <- '~/Data/'
outpath <- '~/Output/'
source('~/Code/functions.R')

# ____________________________________________________________________________ ####
# LOAD DATA ####

## Save sample
load(paste0(inpath, 'MDATA4.RData'))
load(paste0(inpath, 'WB4.RData'))

## Sampled pt and wb
wbt <- wb1 + wb2 + wb3 + wb4
ptt <- pt1 + pt2 + pt3 + pt4

# ____________________________________________________________________________ ####
# ESTIMATION DATA ####

## Sort main data
DATA <- DATA[order(DATA$t, DATA$firm, DATA$serv, DATA$posp), ]

## Price
p <- DATA$p

## Shares
S  <- DATA$s
x  <- aggregate(DATA$s, by = list(DATA$t), function(x) sum(x, na.rm = T))
So <- 1 - x$x[match(DATA$t, x$Group.1)]

## Observables
t  <- DATA$t
X  <- cbind('im' = 1*(DATA$serv == 2), 'pk' = 1*(DATA$serv == 3), 'posp' = DATA$posp)
Xd <- cbind('vm' = 1*(DATA$serv == 1), 'im' = 1*(DATA$serv == 2), 
            'pk' = 1*(DATA$serv == 3), 'posp' = DATA$posp)

## Brands and market effects
dx <- data.frame(p, XB = paste0('_', DATA$j), 
                 XF = paste0('_', DATA$firm),
                 XT = paste0('_', DATA$ANNO),
                 Xt = paste0('_', DATA$t))
XB <- model.matrix(lm(p ~ XB, data = dx))[, -1]
XF <- model.matrix(lm(p ~ XF, data = dx))[, -1]
XT <- model.matrix(lm(p ~ XT, data = dx))[, -1]
Xt <- model.matrix(lm(p ~ Xt, data = dx))[, -1]

## Initials
J      <- nrow(DATA)
NS     <- ncol(v)
mkti   <- match(DATA$t, unique(DATA$t))
delta0 <- log(S) - log(So)

## Firm matrix
Tn <- length(unique(DATA$t))
H  <- matrix(0, ncol = nrow(DATA), nrow = nrow(DATA))
m  <- unique(DATA$t)
for(tt in 1:Tn) {
  x <- cbind('XF_Claro' = 1 - rowSums(XF[t == m[tt],]), XF[t == m[tt],])
  x <- x %*% t(x)
  i <- which(mkti == tt)
  H[i, i] <- x
}

# ____________________________________________________________________________ ####
# ESTIMATE ####

## Instruments
ivn  <- paste0('z', 1:4)
i    <- c(1:4)
Z    <- as.matrix(DATA[, ivn[i]]); c(ncol(Z), rankMatrix(Z)[1])
Z    <- cbind(Z, 1, X, XF, Xt); c(ncol(Z), rankMatrix(Z)[1])
A    <- solve(t(Z) %*% Z, tol = 1e-30)
XPF  <- cbind('constant' = 1, X, XF, XT); ivreg(delta0 ~ 0 + XPF, ~ Z)
WPF  <- cbind('constant' = 1, X, XF, XT)
ZPZ  <- Z %*% A %*% t(Z)
ZPZ2 <- rbind(Z, Z) %*% A %*% t(rbind(Z, Z))
XZPZ <- solve(t(XPF) %*% ZPZ %*% XPF) %*% t(XPF) %*% ZPZ
WZPZ <- solve(t(WPF) %*% ZPZ %*% WPF) %*% t(WPF) %*% ZPZ

# Check identification range
load(paste0(outpath, 'Estim_04_00.RData'))
zeta0 <- cons(c(ans1$par, -0.1))[1:11]
rankMatrix(Z)[1] >= ncol(XPF) + ncol(WPF) + length(zeta0) 

## First stage
ans1 <- optim(zeta0, gmm, method = "Nelder-Mead", control = list(maxit = 10000))
save(ans1, file = paste0(outpath, 'Estim_04_01.RData'))

load(paste0(outpath, 'Estim_04_01.RData'))
zeta <- cons(ans1$par)
out  <- struct(zeta)
b    <- out$b; b
ai   <- -(zeta[11] + zeta[9]*wbt + zeta[10]*v); summary(as.numeric(ai))
cm   <- out$cm; summary(cm); summary(p - cm)

## First-Stage Results
zeta1 <- cons(ans1$par)
om    <- c(out$xi, out$wo)
A2    <- 0
for(i in 1:nrow(Z)) A2 <- A2 + om[i]^2 * Z[i, ]%*%t(Z[i, ])/(nrow(Z)*3)
for(i in 1:nrow(Z)) A2 <- A2 + om[i + nrow(Z)]^2 * Z[i, ]%*%t(Z[i, ])/(nrow(Z)*3)
A2    <- solve(A2, tol = 1e-30)
ZPZ   <- Z %*% A2 %*% t(Z)
ZPZ2  <- rbind(Z, Z) %*% A2 %*% t(rbind(Z, Z))

## Second Stage Estimation
ans  <- optim(zeta1, gmm, method = "Nelder-Mead", control = list(maxit = 15000))
save(ans, file = paste0(outpath, 'Estim_04_02.RData'))

## Results
load(paste0(outpath, 'Estim_04_02.RData'))
zeta  <- cons(ans$par)
out   <- struct(zeta)
delta <- out$delta
cm    <- out$cm; summary(cbind('cm' = cm, 'mkp1' = p - cm))
si    <- out$si
b     <- out$b; b
ai    <- -(zeta[11] + zeta[9]*wbt + zeta[10]*v); summary(as.numeric(ai))
g     <- out$g; g
xi    <- out$xi
wo    <- out$wo
save(list = c('cm', 'delta', 'zeta'), file = paste0(outpath, 'Estimout_04.RData'))

# ____________________________________________________________________________ ####
# VARIANCE ####

## Compute jacobian and moment covariance 
om       <- c(xi, wo)
Z2       <- rbind(Z, Z)
theta    <- c(zeta, b)
eps      <- 1e-13
Kg       <- ncol(Z)
Kb       <- length(theta)
dtheta   <- numeric(Kb)
G   <- PSI <- 0
OM1 <- OM2 <- matrix(0, length(om), Kb)
for(k in 1:Kb) {
  dtheta[k] <- eps
  if(k <= length(zeta)) {
    om1 <- omega2(zeta = zeta - dtheta[1:length(zeta)])[1:nrow(Z2)]
    om2 <- omega2(zeta = zeta + dtheta[1:length(zeta)])[1:nrow(Z2)]
  } else {
    om1 <- omega1(theta[-(1:length(zeta))] - dtheta[(length(zeta) + 1):Kb])[1:nrow(Z2)]
    om2 <- omega1(theta[-(1:length(zeta))] + dtheta[(length(zeta) + 1):Kb])[1:nrow(Z2)]
  }
  dtheta  <- numeric(Kb)
  OM1[, k] <- om1
  OM2[, k] <- om2
}

for(i in 1:nrow(Z2)) {
  print(i)
  # Jacobian (Kg x Kb)
  Gi <- matrix(0, Kg, Kb)
  for(k in 1:Kb) {
    g0 <- Z2[i, ] * om[i]
    g1 <- Z2[i, ] * OM1[i, k]
    g2 <- Z2[i, ] * OM2[i, k]
    Gi[, k] <- 0.5*(g1 - g0)/(-eps) + (g2 - g0)*0.5/eps
  }
  G   <- G + Gi/nrow(Z2) 
  # Moment covariance (Kg x Kg)
  PSI <- PSI + om[i]^2 * Z2[i, ]%*%t(Z2[i, ])/nrow(Z2)
}

## Weight matrix (Kg x Kg)
W <- A2
## Parameter covariance matrix
A   <- t(G) %*% W %*% G
A   <- solve(A, tol = 1e-20)
B   <- t(G) %*% W %*% PSI %*% W %*% G
Var <- (A %*% B %*% A) / (J - Kb)
save(Var, file = paste0(outpath, 'VCOVP_04.RData'))

# ____________________________________________________________________________ ####
# STD. ERRORS & INFERENCE ####

## Create table
load(paste0(outpath, 'VCOVP_04.RData'))
se <- sqrt((diag(Var)))
bb <- c(zeta, b)
Kb <- length(bb)
sb <- tail(se, length(bb))
pv <- (1 - pt(abs(bb/sb), df = J - Kb))/2
x  <- cbind(bb, sb, pv)

## Price parameters
xr <- x[c(11, 9, 10), ]
rownames(xr) <- c('alpha', 'psi', 'sigma')
print(xtable(xr, digits = 3))
## Portability parameters
xr <- x[c(5:8), ]
rownames(xr) <- paste0('phi', 1:4)
print(xtable(xr, digits = 3))
## Winback parameters
xr <- x[c(1:4), ]
rownames(xr) <- paste0('gamma', 1:4)
print(xtable(xr, digits = 3))
## Other parameters
xr <- x[c(12:15), ]
rownames(xr) <- paste0('beta-', c('cosntante', 'internet', 'v+i', 'pospago'))
print(xtable(xr, digits = 3))

# ____________________________________________________________________________ ####
# END ####